// assets/js/signature.js

document.addEventListener('DOMContentLoaded', function() {
    const canvas = document.getElementById('signature-canvas');
    if (!canvas) return; // Exit if the form isn't present on the page
    const ctx = canvas.getContext('2d');
    const signaturePad = document.getElementById('signature-pad');

    // Resize canvas to support high-DPI devices and responsive layouts
    function resizeCanvas() {
        const rect = canvas.getBoundingClientRect();
        const scale = window.devicePixelRatio || 1;
        canvas.width = rect.width * scale;
        canvas.height = rect.height * scale;
        ctx.setTransform(1, 0, 0, 1, 0, 0);
        ctx.scale(scale, scale);
    }
    resizeCanvas();
    window.addEventListener('resize', resizeCanvas);

    let painting = false;
    let lastX = 0;
    let lastY = 0;

    function getPos(e) {
        const rect = canvas.getBoundingClientRect();
        let x, y;
        if (e.touches) {
            if (e.touches.length > 1) return null;
            e.preventDefault();
            const touch = e.touches[0] || e.changedTouches[0];
            x = touch.clientX - rect.left;
            y = touch.clientY - rect.top;
        } else {
            x = e.offsetX;
            y = e.offsetY;
        }
        return { x, y };
    }

    function startPosition(e) {
        const pos = getPos(e);
        if (!pos) return;
        painting = true;
        lastX = pos.x;
        lastY = pos.y;
        draw(e);
    }

    function finishedPosition(e) {
        painting = false;
        ctx.beginPath();
    }

    function draw(e) {
        if (!painting) return;
        const pos = getPos(e);
        if (!pos) return;
        ctx.lineWidth = 3;
        ctx.lineCap = 'round';
        ctx.strokeStyle = '#001b55';
        ctx.beginPath();
        ctx.moveTo(lastX, lastY);
        ctx.lineTo(pos.x, pos.y);
        ctx.stroke();
        lastX = pos.x;
        lastY = pos.y;
        document.getElementById('signature-data').value = canvas.toDataURL();
    }

    // Use Pointer Events if supported (unifies mouse & touch)
    if (window.PointerEvent) {
        canvas.addEventListener('pointerdown', startPosition);
        canvas.addEventListener('pointermove', draw);
        canvas.addEventListener('pointerup', finishedPosition);
        canvas.addEventListener('pointercancel', finishedPosition);
    } else {
        canvas.addEventListener('mousedown', startPosition);
        canvas.addEventListener('mousemove', draw);
        canvas.addEventListener('mouseup', finishedPosition);
        canvas.addEventListener('mouseout', finishedPosition);

        canvas.addEventListener('touchstart', startPosition, { passive: false });
        canvas.addEventListener('touchmove', draw, { passive: false });
        canvas.addEventListener('touchend', finishedPosition);
        canvas.addEventListener('touchcancel', finishedPosition);
    }

    // Clear signature
    document.getElementById('clear-signature').addEventListener('click', function () {
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        document.getElementById('signature-data').value = '';
    });

    // Handle form submission via AJAX
    document.getElementById('custom-form').addEventListener('submit', function (e) {
        e.preventDefault();

        const submitButton = document.getElementById('submit-form');
        submitButton.disabled = true;
        submitButton.value = 'Se procesează...';

        setTimeout(() => {
            submitButton.disabled = false;
            submitButton.value = 'Depune';
        }, 6000);

        const formData = new FormData(this);
        formData.append('action', 'formularul230_submit_form');
        formData.append('formularul230_nonce', document.querySelector('input[name="formularul230_nonce"]').value);

        fetch(form230_ajax_obj.ajax_url, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok.');
            }
            return response.json();
        })
        .then(data => {
            const messagesDiv = document.getElementById('form-messages');
            messagesDiv.innerHTML = '';
            const alertDiv = document.createElement('div');
            alertDiv.classList.add('alert');
            if (data.success) {
                alertDiv.classList.add('alert-success');
                alertDiv.textContent = data.message;
                document.getElementById('custom-form').reset();
                ctx.clearRect(0, 0, canvas.width, canvas.height);
                document.getElementById('signature-data').value = '';
            } else {
                alertDiv.classList.add('alert-danger');
                alertDiv.textContent = data.message || 'A apărut o eroare. Vă rugăm să încercați din nou.';
            }
            messagesDiv.appendChild(alertDiv);
            setTimeout(() => {
                alertDiv.remove();
            }, 7000);
        })
        .catch(error => {
            console.error('Error:', error);
            const messagesDiv = document.getElementById('form-messages');
            messagesDiv.innerHTML = '<div class="alert alert-danger" role="alert">A apărut o eroare. Vă rugăm să încercați din nou.</div>';
        });
    });

    // Prevent default touch actions on the signature pad container
    signaturePad.addEventListener('touchstart', function (e) {
        e.preventDefault();
    }, { passive: false });
    signaturePad.addEventListener('touchmove', function (e) {
        e.preventDefault();
    }, { passive: false });

    // Disable touch zooming (pinch-to-zoom) on the canvas
    canvas.addEventListener('wheel', function (e) {
        e.preventDefault();
    }, { passive: false });

    // Ensure the signature pad container behaves correctly for touch devices
    signaturePad.style.overscrollBehavior = 'contain';
    signaturePad.style.touchAction = 'none';
});
