<?php
// Include modular components
require_once FORMULARUL230_PLUGIN_PATH . 'includes/bulk-actions.php';
require_once FORMULARUL230_PLUGIN_PATH . 'includes/search-functionality.php';

// Function to display the admin interface
function formularul230_admin_interface() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'formularul230_submissions';

    // Handle bulk actions
    formularul230_handle_bulk_actions($table_name);

    // Handle search query with nonce verification
    $search_query = '';
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce is safely handled by wp_verify_nonce()
    if (isset($_GET['s']) && isset($_GET['_wpnonce']) && wp_verify_nonce(wp_unslash($_GET['_wpnonce']), 'formularul230_search')) {
        $search_query = sanitize_text_field(wp_unslash($_GET['s']));
    }
    $submissions = formularul230_search_submissions($search_query);
    ?>
    <div class="wrap">
        <h1>Formularul230 Depuneri</h1>

        <!-- Search Form with Nonce -->
        <form method="get" action="">
            <input type="hidden" name="page" value="formularul230-admin">
            <?php wp_nonce_field('formularul230_search', '_wpnonce'); ?>
            <p class="search-box">
                <label class="screen-reader-text" for="search-submit">Caută depuneri:</label>
                <input type="search" id="search-submit" name="s" value="<?php echo esc_attr($search_query); ?>">
                <input type="submit" class="button" value="Caută depuneri">
            </p>
        </form>

        <!-- Combined Bulk Actions and Submissions Table Form -->
        <form method="post" action="">
            <?php wp_nonce_field('formularul230_bulk_action', '_wpnonce'); ?>
            <div class="tablenav top">
                <!-- Bulk Actions Dropdown -->
                <select name="bulk_action">
                    <option value="">Acțiuni în masă</option>
                    <option value="delete">Șterge</option>
                </select>
                <input type="submit" class="button action" value="Aplică">
                <!-- Export and Download Buttons -->
                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=formularul230-admin&formularul230_export=1&export_all=1'), 'formularul230_export_csv')); ?>" class="button button-primary">Exportă CSV</a>       
            </div>
            <!-- Submissions Table -->
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 5%;"><input type="checkbox" id="select-all"></th>
                        <th style="width: 5%;">ID</th>
                        <th>Nume</th>
                        <th style="width: 15%;">Email</th>
                        <th>Telefon</th>
                        <th style="width: 7%;">Oraș</th>
                        <th style="width: 7%;">Județ</th>
                        <th style="width: 10%;">Creat la</th>
                        <th style="width: 15%;">Acțiuni</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($submissions)) : ?>
                        <?php foreach ($submissions as $submission) : ?>
                            <tr>
                                <td style="width: 5%;">
                                    <input type="checkbox" name="submission_ids[]" value="<?php echo esc_attr($submission['id']); ?>">
                                </td>
                                <td style="width: 5%;"><?php echo esc_html($submission['id']); ?></td>
                                <td><?php echo esc_html($submission['prename']) . ' ' . esc_html($submission['name']); ?></td>
                                <td style="width: 15%;">
                                    <?php if (!empty($submission['email'])) : ?>
                                        <?php echo esc_html($submission['email']); ?>
                                    <?php else : ?>
                                        <span class="badge badge-secondary">Fără email</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if (!empty($submission['phone'])) : ?>
                                        <?php echo esc_html(formularul230_format_phone_number($submission['phone'])); ?>
                                    <?php else : ?>
                                        <span class="badge badge-secondary">Fără telefon</span>
                                    <?php endif; ?>
                                </td>
                                <td style="width: 7%;"><?php echo esc_html($submission['city']); ?></td>
                                <td style="width: 7%;"><?php echo esc_html($submission['county']); ?></td>
                                <td style="width: 10%;">
                                    <?php
                                    $created_at = strtotime($submission['created_at']);
                                    $formatted_date = gmdate('d F Y', $created_at);
                                    $formatted_time = gmdate('h:i A', $created_at);
                                    ?>
                                    <span class="badge badge-date">Data: <?php echo esc_html($formatted_date); ?></span><br>
                                    <span class="badge badge-time">Ora: <?php echo esc_html($formatted_time); ?></span>
                                </td>
                                <td style="width: 15%;">
                                    <?php
                                    $preview_submission = wp_create_nonce('formularul230_preview_submission');
                                    $preview_url = add_query_arg(
                                        array(
                                            'page' => 'formularul230-submission-preview',
                                            'id' => $submission['id'],
                                            '_wpnonce' => $preview_submission,
                                        ),
                                        admin_url('admin.php')
                                    );
                                    $delete_nonce = wp_create_nonce('formularul230_delete_nonce');
                                    $delete_url = add_query_arg(
                                        array(
                                            'page' => 'formularul230-admin',
                                            'action' => 'delete',
                                            'id' => $submission['id'],
                                            '_wpnonce' => $delete_nonce,
                                        ),
                                        admin_url('admin.php')
                                    );
                                    ?>
                                    <a href="<?php echo esc_url($preview_url); ?>" class="button button-small">Previzualizează</a>
                                    <a href="<?php echo esc_url($delete_url); ?>" class="button button-small delete-submission">Șterge</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr>
                            <td colspan="9">Nicio depunere existentă!</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </form>
    </div>
    <script>
        // Select all checkboxes
        document.getElementById('select-all').addEventListener('click', function () {
            const checkboxes = document.querySelectorAll('input[name="submission_ids[]"]');
            checkboxes.forEach(checkbox => checkbox.checked = this.checked);
        });

        // SweetAlert2 for Delete Confirmation
        document.addEventListener('DOMContentLoaded', function () {
            const deleteButtons = document.querySelectorAll('.delete-submission');
            deleteButtons.forEach(button => {
                button.addEventListener('click', function (e) {
                    e.preventDefault();
                    const deleteUrl = this.getAttribute('href');
                    Swal.fire({
                        title: 'Sunteți sigur?',
                        text: "Această depunere va fi ștearsă permanent! Acțiunea este ireversibilă!",
                        icon: 'warning',
                        showCancelButton: true,
                        confirmButtonColor: '#3085d6',
                        cancelButtonColor: '#d33',
                        confirmButtonText: 'Da, șterge-l!',
                        cancelButtonText: 'Nu, anulează!'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            window.location.href = deleteUrl;
                        }
                    });
                });
            });
        });
    </script>
    <?php
}

// Add the admin menu page
function formularul230_add_admin_menu() {
    $icon_url = plugins_url('../assets/icons/formularul230_logo4_resized.svg', __FILE__);
    add_menu_page(
        'Formularul230 Submissions',
        'Form230',
        'manage_options',
        'formularul230-admin',
        'formularul230_admin_interface',
        $icon_url
    );

    // Add subpages, renaming the first one to "Formulare"
    add_submenu_page(
        'formularul230-admin',
        __('Formulare', 'Formularul230'),
        __('Formulare', 'Formularul230'),
        'manage_options',
        'formularul230-admin',
        'formularul230_admin_interface'
    );
}
add_action('admin_menu', 'formularul230_add_admin_menu');

/**
 * Helper function to format phone numbers
 */
function formularul230_format_phone_number($phone) {
    if (strlen($phone) === 10) {
        return substr($phone, 0, 4) . ' ' . substr($phone, 4, 3) . ' ' . substr($phone, 7, 3);
    }
    return $phone; // Return as-is if not exactly 10 digits
}